<?php
require_once __DIR__ . '/config.php';

function db(){
  static $pdo;
  if(!$pdo){
    $pdo = new PDO(
      'mysql:host='.DB_HOST.';dbname='.DB_NAME.';charset=utf8mb4',
      DB_USER, DB_PASS,
      [PDO::ATTR_ERRMODE=>PDO::ERRMODE_EXCEPTION,
       PDO::ATTR_DEFAULT_FETCH_MODE=>PDO::FETCH_ASSOC]
    );
  }
  return $pdo;
}

function getv($k, $def=''){ return isset($_GET[$k]) ? trim((string)$_GET[$k]) : $def; }
function postv($k, $def=''){ return isset($_POST[$k]) ? trim((string)$_POST[$k]) : $def; }
function jres($arr){
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

function ensure_token_row($token, $sid='', $ip='', $action='open'){
  $pdo = db();
  $stmt = $pdo->prepare("SELECT token FROM counter_strike_tokens WHERE token=? LIMIT 1");
  $stmt->execute([$token]);
  if(!$stmt->fetch()){
    $stmt = $pdo->prepare("INSERT INTO counter_strike_tokens (token, steamid, ip, action, status, username, remember, created_at, expires_at)
                           VALUES (?, ?, ?, ?, 'PENDING', NULL, 0, NOW(), DATE_ADD(NOW(), INTERVAL 10 MINUTE))");
    $stmt->execute([$token, $sid ?: null, $ip ?: null, $action]);
  }
}

function set_token_status($token, $status, $username=null, $remember=0){
  $pdo = db();
  $stmt = $pdo->prepare("UPDATE counter_strike_tokens SET status=?, username=?, remember=?, updated_at=NOW() WHERE token=?");
  $stmt->execute([$status, $username, (int)$remember, $token]);
}

function find_user_by_username($username){
  $pdo = db();
  $st = $pdo->prepare("SELECT * FROM counter_strike_users_util WHERE username=? LIMIT 1");
  $st->execute([$username]);
  return $st->fetch();
}

function find_user_by_sid_or_ip($sid, $ip){
  $pdo = db();
  $st = $pdo->prepare("SELECT * FROM counter_strike_users_util WHERE (steamid IS NOT NULL AND steamid=?) OR (ip IS NOT NULL AND ip=?) ORDER BY last_login DESC LIMIT 1");
  $st->execute([$sid ?: '', $ip ?: '']);
  return $st->fetch();
}

function create_user($username, $password, $sid='', $ip=''){
  $pdo = db();
  $hash = password_hash($password, PASSWORD_DEFAULT);
  $st = $pdo->prepare("INSERT INTO counter_strike_users_util (username, passhash, steamid, ip, created_at, last_login) VALUES (?,?,?,?,NOW(),NOW())");
  $st->execute([$username, $hash, $sid ?: null, $ip ?: null]);
  return $pdo->lastInsertId();
}

function touch_login($user_id, $sid='', $ip=''){
  $pdo = db();
  $st = $pdo->prepare("UPDATE counter_strike_users_util SET last_login=NOW(), steamid=COALESCE(?, steamid), ip=COALESCE(?, ip) WHERE id=?");
  $st->execute([$sid ?: null, $ip ?: null, $user_id]);
}

/**
 * Update user skin file on CS server (in configs folder)
 * This function creates/updates the user_skins_<username>.txt file
 * 
 * @param string $username Username
 * @param string $configs_dir Path to configs folder (if null, tries to detect)
 * @return bool True on success, false on failure
 */
function update_user_skin_file($username, $configs_dir = null) {
    if (empty($username)) {
        return false;
    }
    
    try {
        $pdo = db();
        
        // Get user ID from counter_strike_users (not counter_strike_users_util)
        $stmt = $pdo->prepare("SELECT id FROM counter_strike_users WHERE username = ? LIMIT 1");
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return false; // User not found
        }
        
        $user_id = intval($user['id']);
        
        // Get user's skins from database
        $stmt = $pdo->prepare("SELECT skin_name FROM counter_strike_user_skins WHERE user_id = ? ORDER BY skin_name ASC");
        $stmt->execute([$user_id]);
        $skins = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // Determine configs directory
        if ($configs_dir === null) {
            // Try to detect: if this script is in PE HOST, configs is one level up
            $pe_host_dir = __DIR__;
            $configs_dir = dirname($pe_host_dir) . '/configs';
            
            // If that doesn't exist, try common paths
            if (!is_dir($configs_dir)) {
                // Try absolute path (adjust if needed)
                $possible_paths = [
                    'D:\\cs_server\\cstrike\\addons\\amxmodx\\configs',
                    'C:\\hlserver\\cstrike\\addons\\amxmodx\\configs',
                ];
                
                foreach ($possible_paths as $path) {
                    if (is_dir($path)) {
                        $configs_dir = $path;
                        break;
                    }
                }
            }
        }
        
        if (!is_dir($configs_dir)) {
            // Cannot write - configs directory not accessible
            // This is OK if PHP is on different server than CS server
            // Plugin will create file when user connects
            return false;
        }
        
        // Create file
        $filename = $configs_dir . '/user_skins_' . $username . '.txt';
        $fp = @fopen($filename, 'w');
        
        if (!$fp) {
            return false; // Cannot write file
        }
        
        fwrite($fp, "OK\n");
        foreach ($skins as $skin) {
            fwrite($fp, trim($skin) . "\n");
        }
        fclose($fp);
        
        return true;
    } catch (Exception $e) {
        // Error - log if possible, but don't break the purchase flow
        error_log("update_user_skin_file error for $username: " . $e->getMessage());
        return false;
    }
}
