<?php
header('Content-Type: text/html; charset=windows-1252');
header('X-Frame-Options: ALLOWALL');
header('Content-Security-Policy: frame-ancestors *');

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/util.php';

function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'ISO-8859-1'); }

// Check if user is logged in (from counter_strike_users table)
// Verification by username - if user is logged in on server with same name, they can shop
$user_id = null;
$user_money = 0;
$user_logged_in = false;
$username = '';

// Get username from GET parameter (sent by plugin when opening shop)
$username = trim($_GET['username'] ?? '');

if (!empty($username)) {
    try {
        $pdo = db();
        // Check if user exists and has auto_login active (logged in on server)
        // Also check if last_login was recent (within last 24 hours) to ensure they're still active
        $stmt = $pdo->prepare("SELECT id, money, auto_login, last_login FROM counter_strike_users WHERE username = ? LIMIT 1");
        $stmt->execute([$username]);
        $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user_data) {
            // User is considered logged in if:
            // 1. They have auto_login enabled (logged in on server)
            // 2. They logged in recently (within last 24 hours)
            $last_login = $user_data['last_login'];
            $is_recent = false;
            if ($last_login) {
                $last_login_time = strtotime($last_login);
                $is_recent = (time() - $last_login_time) < (24 * 60 * 60); // 24 hours
            }
            
            if ($user_data['auto_login'] == 1 && $is_recent) {
                $user_id = intval($user_data['id']);
                $user_money = floatval($user_data['money']);
                $user_logged_in = true;
            }
        }
    } catch (Exception $e) {
        // Error checking user
    }
}

// Handle purchase
$purchase_msg = '';
$purchase_error = '';
if ($user_logged_in && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['buy_skin'])) {
    $skin_id = intval($_POST['skin_id'] ?? 0);
    
    // Verify username is still provided (from hidden field in form)
    $verify_username = trim($_POST['username'] ?? $username);
    
    if ($skin_id > 0 && !empty($verify_username)) {
        try {
            $pdo = db();
            $pdo->beginTransaction();
            
            // Get skin info
            $stmt = $pdo->prepare("SELECT id, name, price FROM counter_strike_skins WHERE id = ? AND active = 1");
            $stmt->execute([$skin_id]);
            $skin = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$skin) {
                throw new Exception('Skin not found or not available');
            }
            
            // Verify user is still logged in and get user_id
            $stmt = $pdo->prepare("SELECT id, money FROM counter_strike_users WHERE username = ? AND auto_login = 1 FOR UPDATE");
            $stmt->execute([$verify_username]);
            $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user_data) {
                throw new Exception('User not found or not logged in');
            }
            
            $verify_user_id = intval($user_data['id']);
            
            // Check if user already has this skin
            $stmt = $pdo->prepare("SELECT id FROM counter_strike_user_skins WHERE user_id = ? AND skin_name = ?");
            $stmt->execute([$verify_user_id, $skin['name']]);
            if ($stmt->fetch()) {
                throw new Exception('You already own this skin');
            }
            
            $current_money = floatval($user_data['money']);
            $skin_price = floatval($skin['price']);
            
            if ($current_money < $skin_price) {
                throw new Exception('Not enough money');
            }
            
            // Deduct money
            $new_money = $current_money - $skin_price;
            $stmt = $pdo->prepare("UPDATE counter_strike_users SET money = ? WHERE id = ?");
            $stmt->execute([$new_money, $verify_user_id]);
            
            // Add skin to user
            $stmt = $pdo->prepare("INSERT INTO counter_strike_user_skins (user_id, skin_name) VALUES (?, ?)");
            $stmt->execute([$verify_user_id, $skin['name']]);
            
            $pdo->commit();
            $purchase_msg = 'Skin purchased successfully!';
            $user_money = $new_money;
            $user_id = $verify_user_id; // Update for display
            
            // Update user skin file automatically (if configs folder is accessible)
            update_user_skin_file($verify_username);
        } catch (Exception $e) {
            $pdo->rollBack();
            $purchase_error = $e->getMessage();
        }
    }
}

// Get all active skins
$skins = [];
try {
    $pdo = db();
    $stmt = $pdo->query("SELECT * FROM counter_strike_skins WHERE active = 1 ORDER BY created_at DESC");
    $skins = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get user's purchased skins if logged in
    $user_skins = [];
    if ($user_logged_in) {
        $stmt = $pdo->prepare("SELECT skin_name FROM counter_strike_user_skins WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $user_skins = $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    // Mark which skins user owns
    foreach ($skins as &$skin) {
        $skin['owned'] = in_array($skin['name'], $user_skins);
    }
    unset($skin);
} catch (Exception $e) {
    $error = 'Error loading skins: ' . $e->getMessage();
}
?>
<!DOCTYPE html>
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=windows-1252">
<title>Skins Shop</title>
<style>
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

/* MOTD-compatible scroll - body scrolls, not inner container */
html, body {
    height: auto;
    min-height: 100%;
    overflow-x: hidden;
    overflow-y: auto;
}

body {
    background: #0b0b0b;
    color: #eee;
    font-family: Arial, Helvetica, sans-serif;
    margin: 0;
    padding: 0;
}

/* Hide scrollbar visually but keep functionality */
body::-webkit-scrollbar {
    width: 12px;
}

body::-webkit-scrollbar-track {
    background: #0b0b0b;
}

body::-webkit-scrollbar-thumb {
    background: #8bc34a;
    border: 2px solid #4caf50;
}

body::-webkit-scrollbar-thumb:hover {
    background: #4caf50;
}

.wrap {
    max-width: 900px;
    margin: 0 auto;
    padding: 20px;
    background: #111;
    border: 1px solid #222;
    min-height: 100%;
}

h3 {
    margin: 0 0 20px;
    text-align: center;
    color: #8bc34a;
    font-size: 24px;
    font-weight: bold;
    text-transform: uppercase;
}

.user-info {
    text-align: center;
    padding: 15px;
    margin-bottom: 20px;
    background: #181818;
    border: 1px solid #333;
}

.user-info.logged-in {
    border-color: #8bc34a;
}

.user-info .money {
    color: #4caf50;
    font-size: 18px;
    font-weight: bold;
    margin-top: 5px;
}

.user-info .not-logged {
    color: #f44336;
    font-size: 14px;
}

.msg {
    padding: 12px;
    margin: 15px 0;
    border-radius: 4px;
    text-align: center;
    font-weight: bold;
}

.msg.success {
    background: rgba(139, 195, 74, 0.2);
    border: 1px solid #8bc34a;
    color: #8bc34a;
}

.msg.error {
    background: rgba(244, 67, 54, 0.2);
    border: 1px solid #f44336;
    color: #f44336;
}

/* Shop container - NO fixed height, lets body scroll */
.shop-container {
    padding: 10px 0;
    margin-top: 10px;
    width: 100%;
    /* Remove overflow and max-height - let body handle scrolling */
}

.shop-grid {
    width: 100%;
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    justify-content: flex-start;
}

.shop-item {
    width: calc(33.333% - 10px);
    min-width: 200px;
    background: #181818;
    border: 2px solid #8bc34a;
    padding: 10px;
    text-align: center;
    transition: all 0.3s ease;
    box-sizing: border-box;
}

.shop-item:hover {
    border-color: #4caf50;
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(139, 195, 74, 0.3);
}

.shop-item.owned {
    border-color: #666;
    opacity: 0.7;
}

.shop-item img {
    width: 100%;
    height: 180px;
    object-fit: cover;
    display: block;
    background: #0b0b0b;
    margin-bottom: 10px;
    border: 1px solid #333;
}

.shop-item-name {
    color: #8bc34a;
    font-weight: bold;
    font-size: 14px;
    margin-bottom: 8px;
    min-height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.shop-item-price {
    color: #4caf50;
    font-size: 16px;
    font-weight: bold;
    margin-bottom: 10px;
}

.shop-item-owned {
    color: #666;
    font-size: 12px;
    margin-bottom: 10px;
    font-style: italic;
}

.buy-btn {
    width: 100%;
    padding: 10px;
    background: #8bc34a;
    border: 2px solid #4caf50;
    color: #fff;
    font-size: 14px;
    font-weight: bold;
    cursor: pointer;
    text-transform: uppercase;
    transition: all 0.3s ease;
}

.buy-btn:hover:not(:disabled) {
    background: #4caf50;
    transform: scale(1.05);
}

.buy-btn:disabled {
    background: #666;
    border-color: #555;
    cursor: not-allowed;
    opacity: 0.6;
}

/* Responsive - for smaller MOTD windows */
@media (max-width: 768px) {
    .shop-item {
        width: calc(50% - 8px);
    }
}

@media (max-width: 480px) {
    .shop-item {
        width: 100%;
    }
    
    .wrap {
        padding: 15px;
    }
}
</style>
</head>
<body>
<div class="wrap">
  <h3>Skins Shop</h3>
  
  <!-- User Info -->
  <div class="user-info <?= $user_logged_in ? 'logged-in' : '' ?>">
    <?php if ($user_logged_in): ?>
      <div>Logged in</div>
      <div class="money">Money: <?= number_format($user_money, 2) ?> servercoins</div>
    <?php else: ?>
      <div class="not-logged">You must be logged in to purchase skins</div>
    <?php endif; ?>
  </div>
  
  <!-- Messages -->
  <?php if ($purchase_msg): ?>
    <div class="msg success"><?= h($purchase_msg) ?></div>
  <?php endif; ?>
  <?php if ($purchase_error): ?>
    <div class="msg error"><?= h($purchase_error) ?></div>
  <?php endif; ?>
  
  <!-- Shop Grid -->
  <div class="shop-container">
    <div class="shop-grid">
      <?php if (empty($skins)): ?>
        <div style="width: 100%; text-align: center; color: #666; padding: 40px;">
          No skins available in the shop.
        </div>
      <?php else: ?>
        <?php foreach ($skins as $skin): ?>
          <div class="shop-item <?= $skin['owned'] ? 'owned' : '' ?>">
            <img src="<?= h($skin['image_url']) ?>" alt="<?= h($skin['name']) ?>" 
                 onerror="this.src='data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'200\' height=\'180\'%3E%3Crect fill=\'%23333\' width=\'200\' height=\'180\'/%3E%3Ctext x=\'50%25\' y=\'50%25\' text-anchor=\'middle\' dy=\'.3em\' fill=\'%23666\' font-size=\'14\'%3ENo Image%3C/text%3E%3C/svg%3E'">
            <div class="shop-item-name"><?= h($skin['name']) ?></div>
            <div class="shop-item-price"><?= number_format($skin['price'], 2) ?> servercoins</div>
            
            <?php if ($skin['owned']): ?>
              <div class="shop-item-owned">Already Owned</div>
              <button class="buy-btn" disabled>Owned</button>
            <?php else: ?>
              <form method="post" style="margin: 0;">
                <input type="hidden" name="skin_id" value="<?= $skin['id'] ?>">
                <input type="hidden" name="username" value="<?= h($username) ?>">
                <button type="submit" name="buy_skin" class="buy-btn" 
                        <?= !$user_logged_in || $user_money < $skin['price'] ? 'disabled' : '' ?>>
                  <?= !$user_logged_in ? 'Login Required' : ($user_money < $skin['price'] ? 'Not Enough Money' : 'Buy') ?>
                </button>
              </form>
            <?php endif; ?>
          </div>
        <?php endforeach; ?>
      <?php endif; ?>
    </div>
  </div>
</div>
</body>
</html>

