<?php
header('Content-Type: text/html; charset=utf-8');
header('X-Frame-Options: ALLOWALL');
header('Content-Security-Policy: frame-ancestors *');

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/util.php';

function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function normalize_video_url($url) {
    $url = trim($url ?? '');
    if ($url === '') {
        return '';
    }

    $lower = strtolower($url);

    // Convert common YouTube links (watch, share, shorts) to embeddable URLs
    if (strpos($lower, 'youtube.com') !== false || strpos($lower, 'youtu.be') !== false) {
        if (preg_match('~(?:youtube\.com/(?:watch\?v=|embed/|shorts/)|youtu\.be/)([A-Za-z0-9_-]{11})~', $url, $match)) {
            return 'https://www.youtube.com/embed/' . $match[1];
        }
    }

    return $url;
}

function is_inline_video_file($url) {
    if ($url === '') {
        return false;
    }

    $path = parse_url($url, PHP_URL_PATH);
    if (!$path) {
        return false;
    }

    return (bool)preg_match('/\.(mp4|webm|ogg)(\?.*)?$/i', $path);
}

// Table name following the same pattern as other CS tables
define('PLUGINS_TABLE', 'counter_strike_plugin_presentations');

$plugin = null;
$plugin_id = intval($_GET['id'] ?? 0);

if ($plugin_id > 0) {
    try {
        $pdo = db();
        $stmt = $pdo->prepare("SELECT * FROM " . PLUGINS_TABLE . " WHERE id = ? AND active = 1");
        $stmt->execute([$plugin_id]);
        $plugin = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        // Error loading plugin
    }
}

if (!$plugin) {
    header('Location: plugins_gallery.php');
    exit;
}

// Parse gallery images
$gallery_images = [];
if (!empty($plugin['gallery_images'])) {
    $gallery_images = array_map('trim', explode(',', $plugin['gallery_images']));
    $gallery_images = array_filter($gallery_images);
}

$raw_video_url = trim($plugin['video_url'] ?? '');
$normalized_video_url = normalize_video_url($raw_video_url);
$is_inline_video = is_inline_video_file($normalized_video_url);
?>
<!DOCTYPE html>
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=utf-8">
<title><?= h($plugin['name']) ?> - Plugin Details</title>
<style>
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

html, body {
    height: auto;
    min-height: 100%;
    overflow-x: hidden;
    overflow-y: auto;
}

body {
    background: #0b0b0b;
    color: #eee;
    font-family: Arial, Helvetica, sans-serif;
    margin: 0;
    padding: 0;
}

body::-webkit-scrollbar {
    width: 12px;
}

body::-webkit-scrollbar-track {
    background: #0b0b0b;
}

body::-webkit-scrollbar-thumb {
    background: #8bc34a;
    border: 2px solid #4caf50;
}

body::-webkit-scrollbar-thumb:hover {
    background: #4caf50;
}

.wrap {
    max-width: 1000px;
    margin: 0 auto;
    padding: 20px;
    background: #111;
    border: 1px solid #222;
    min-height: 100%;
}

.back-link {
    display: inline-block;
    margin-bottom: 20px;
    color: #8bc34a;
    text-decoration: none;
    font-size: 14px;
    transition: color 0.3s;
}

.back-link:hover {
    color: #4caf50;
}

h1 {
    margin: 0 0 30px;
    color: #8bc34a;
    font-size: 32px;
    font-weight: bold;
    text-align: center;
}

.plugin-header {
    text-align: center;
    margin-bottom: 40px;
}

.plugin-header img {
    max-width: 100%;
    max-height: 400px;
    border: 2px solid #8bc34a;
    margin-bottom: 20px;
}

.section {
    margin-bottom: 40px;
    padding: 20px;
    background: #181818;
    border: 1px solid #333;
}

.section h2 {
    color: #8bc34a;
    font-size: 20px;
    margin-bottom: 15px;
    border-bottom: 2px solid #8bc34a;
    padding-bottom: 10px;
}

.section p, .section div {
    color: #ccc;
    line-height: 1.6;
    font-size: 14px;
}

.section pre {
    background: #0b0b0b;
    padding: 15px;
    border: 1px solid #333;
    overflow-x: auto;
    color: #8bc34a;
    font-size: 13px;
    white-space: pre-wrap;
    word-wrap: break-word;
}

.video-container {
    position: relative;
    width: 100%;
    padding-bottom: 56.25%; /* 16:9 aspect ratio */
    height: 0;
    overflow: hidden;
    background: #0b0b0b;
    border: 2px solid #8bc34a;
}

.video-container video, .video-container iframe {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
}

.gallery-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-top: 15px;
}

.gallery-grid img {
    width: 100%;
    height: 200px;
    object-fit: cover;
    border: 2px solid #8bc34a;
    cursor: pointer;
    transition: transform 0.3s;
}

.gallery-grid img:hover {
    transform: scale(1.05);
}

.download-section {
    margin: 40px 0;
    text-align: center;
}

.download-btn {
    display: inline-block;
    padding: 18px 36px;
    font-size: 18px;
    font-weight: bold;
    color: #0b0b0b;
    background: linear-gradient(135deg, #8bc34a, #4caf50);
    border: none;
    border-radius: 6px;
    text-decoration: none;
    text-transform: uppercase;
    letter-spacing: 1px;
    transition: transform 0.2s, box-shadow 0.2s;
    box-shadow: 0 6px 15px rgba(76, 175, 80, 0.35);
}

.download-btn:hover {
    transform: translateY(-2px) scale(1.01);
    box-shadow: 0 10px 18px rgba(139, 195, 74, 0.45);
}

.download-btn:active {
    transform: translateY(0);
}

@media (max-width: 768px) {
    .gallery-grid {
        grid-template-columns: repeat(2, 1fr);
    }
}
</style>
</head>
<body>
<div class="wrap">
  <a href="plugins_gallery.php" class="back-link">← Back to Plugins</a>
  
  <h1><?= h($plugin['name']) ?></h1>
  
  <?php if ($plugin['image_url']): ?>
    <div class="plugin-header">
      <img src="<?= h($plugin['image_url']) ?>" alt="<?= h($plugin['name']) ?>">
    </div>
  <?php endif; ?>
  
  <?php if ($plugin['description']): ?>
    <div class="section">
      <h2>Description</h2>
      <p><?= nl2br(h($plugin['description'])) ?></p>
    </div>
  <?php endif; ?>
  
  <?php if ($plugin['installation_tutorial']): ?>
    <div class="section">
      <h2>Installation Tutorial</h2>
      <pre><?= h($plugin['installation_tutorial']) ?></pre>
    </div>
  <?php endif; ?>
  
  <?php if ($normalized_video_url): ?>
    <div class="section">
      <h2>Video</h2>
      <div class="video-container">
        <?php if ($is_inline_video): ?>
          <video controls>
            <source src="<?= h($normalized_video_url) ?>" type="video/mp4">
            Your browser does not support the video tag.
          </video>
        <?php else: ?>
          <iframe 
            src="<?= h($normalized_video_url) ?>" 
            frameborder="0" 
            allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" 
            allowfullscreen>
          </iframe>
        <?php endif; ?>
      </div>
    </div>
  <?php endif; ?>
  
  <?php if (!empty($gallery_images)): ?>
    <div class="section">
      <h2>Gallery</h2>
      <div class="gallery-grid">
        <?php foreach ($gallery_images as $img_url): ?>
          <img src="<?= h(trim($img_url)) ?>" alt="Gallery image" onclick="window.open(this.src, '_blank')">
        <?php endforeach; ?>
      </div>
    </div>
  <?php endif; ?>

  <?php if (!empty($plugin['download_url'])): ?>
    <div class="download-section">
      <a href="<?= h($plugin['download_url']) ?>" class="download-btn" target="_blank" rel="noopener">Download Plugin</a>
    </div>
  <?php endif; ?>
</div>
</body>
</html>

