<?php
// Admin panel for plugins (NO LOGIN REQUIRED - hidden from public)
header('Content-Type: text/html; charset=utf-8');
header('X-Frame-Options: ALLOWALL');
header('Content-Security-Policy: frame-ancestors *');

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/util.php';

function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function redirect_with(array $params = [])
{
    $target = 'admin_plugins.php';
    if (!empty($params)) {
        $target .= '?' . http_build_query($params);
    }
    header('Location: ' . $target);
    exit;
}

// Table name following the same pattern as other CS tables
define('PLUGINS_TABLE', 'counter_strike_plugin_presentations');

// Create plugins table if it doesn't exist
try {
    $pdo = db();
    // Check if table exists and has correct structure
    $stmt = $pdo->query("SHOW TABLES LIKE '" . PLUGINS_TABLE . "'");
    $table_exists = $stmt->rowCount() > 0;
    
    if (!$table_exists) {
        // Create new table with correct structure
        $pdo->exec("CREATE TABLE " . PLUGINS_TABLE . " (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            image_url VARCHAR(500),
            description TEXT,
            installation_tutorial TEXT,
            video_url VARCHAR(500),
            gallery_images TEXT,
            download_url VARCHAR(500),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            active TINYINT(1) DEFAULT 1
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
    } else {
        // Table exists - check and fix column types if needed
        $stmt = $pdo->query("SHOW COLUMNS FROM " . PLUGINS_TABLE . " LIKE 'installation_tutorial'");
        $col = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($col && (stripos($col['Type'], 'timestamp') !== false || stripos($col['Type'], 'datetime') !== false)) {
            // Fix wrong column type
            $pdo->exec("ALTER TABLE " . PLUGINS_TABLE . " MODIFY installation_tutorial TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        }
        
        // Also check and fix other text columns
        $text_columns = ['description', 'gallery_images'];
        foreach ($text_columns as $col_name) {
            $stmt = $pdo->query("SHOW COLUMNS FROM " . PLUGINS_TABLE . " LIKE '$col_name'");
            $col = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($col && (stripos($col['Type'], 'timestamp') !== false || stripos($col['Type'], 'datetime') !== false)) {
                $pdo->exec("ALTER TABLE " . PLUGINS_TABLE . " MODIFY $col_name TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            }
        }
        
        // Ensure download_url column exists
        $stmt = $pdo->query("SHOW COLUMNS FROM " . PLUGINS_TABLE . " LIKE 'download_url'");
        if ($stmt->rowCount() === 0) {
            $pdo->exec("ALTER TABLE " . PLUGINS_TABLE . " ADD download_url VARCHAR(500) NULL AFTER gallery_images");
        }

        // Ensure table uses utf8mb4
        $pdo->exec("ALTER TABLE " . PLUGINS_TABLE . " CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    }
} catch (Exception $e) {
    // Table creation/update error - log but don't break
    error_log("Plugin table error: " . $e->getMessage());
}

$success_msg = '';
$error_msg = '';
$status = $_GET['status'] ?? '';

if ($status === 'added') {
    $success_msg = 'Plugin added successfully!';
} elseif ($status === 'updated') {
    $success_msg = 'Plugin updated successfully!';
} elseif ($status === 'deleted') {
    $success_msg = 'Plugin deleted successfully!';
} elseif ($status === 'toggled') {
    $success_msg = 'Plugin status updated!';
}

// Handle add plugin
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_plugin') {
    $name = trim($_POST['name'] ?? '');
    $image_url = trim($_POST['image_url'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $installation_tutorial = trim($_POST['installation_tutorial'] ?? '');
    $video_url = trim($_POST['video_url'] ?? '');
    $gallery_images = trim($_POST['gallery_images'] ?? ''); // Comma-separated URLs
    $download_url = trim($_POST['download_url'] ?? '');
    
    if ($name) {
        try {
            $pdo = db();
            // Ensure UTF-8 encoding for text fields
            $pdo->exec("SET NAMES utf8mb4");
            $stmt = $pdo->prepare("INSERT INTO " . PLUGINS_TABLE . " (name, image_url, description, installation_tutorial, video_url, gallery_images, download_url) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $name, 
                $image_url ?: null, 
                $description ?: null, 
                $installation_tutorial ?: null, 
                $video_url ?: null, 
                $gallery_images ?: null,
                $download_url ?: null
            ]);
            redirect_with(['status' => 'added']);
        } catch (Exception $e) {
            $error_msg = 'Error: ' . $e->getMessage();
        }
    } else {
        $error_msg = 'Plugin name is required';
    }
}

// Handle update plugin
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_plugin') {
    $id = intval($_POST['id'] ?? 0);
    $name = trim($_POST['name'] ?? '');
    $image_url = trim($_POST['image_url'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $installation_tutorial = trim($_POST['installation_tutorial'] ?? '');
    $video_url = trim($_POST['video_url'] ?? '');
    $gallery_images = trim($_POST['gallery_images'] ?? '');
    $download_url = trim($_POST['download_url'] ?? '');
    
    if ($id > 0 && $name) {
        try {
            $pdo = db();
            // Ensure UTF-8 encoding for text fields
            $pdo->exec("SET NAMES utf8mb4");
            $stmt = $pdo->prepare("UPDATE " . PLUGINS_TABLE . " SET name = ?, image_url = ?, description = ?, installation_tutorial = ?, video_url = ?, gallery_images = ?, download_url = ? WHERE id = ?");
            $stmt->execute([
                $name, 
                $image_url ?: null, 
                $description ?: null, 
                $installation_tutorial ?: null, 
                $video_url ?: null, 
                $gallery_images ?: null,
                $download_url ?: null,
                $id
            ]);
            redirect_with(['status' => 'updated', 'edit' => $id]);
        } catch (Exception $e) {
            $error_msg = 'Error: ' . $e->getMessage();
        }
    }
}

// Handle delete plugin
if (isset($_GET['delete'])) {
    try {
        $pdo = db();
        $stmt = $pdo->prepare("DELETE FROM " . PLUGINS_TABLE . " WHERE id = ?");
        $stmt->execute([intval($_GET['delete'])]);
        redirect_with(['status' => 'deleted']);
    } catch (Exception $e) {
        $error_msg = 'Error: ' . $e->getMessage();
    }
}

// Handle toggle active
if (isset($_GET['toggle'])) {
    try {
        $pdo = db();
        $stmt = $pdo->prepare("UPDATE " . PLUGINS_TABLE . " SET active = NOT active WHERE id = ?");
        $stmt->execute([intval($_GET['toggle'])]);
        redirect_with(['status' => 'toggled']);
    } catch (Exception $e) {
        $error_msg = 'Error: ' . $e->getMessage();
    }
}

// Get all plugins
$plugins = [];
$editing_plugin = null;
if (isset($_GET['edit'])) {
    try {
        $pdo = db();
        $stmt = $pdo->prepare("SELECT * FROM " . PLUGINS_TABLE . " WHERE id = ?");
        $stmt->execute([intval($_GET['edit'])]);
        $editing_plugin = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        $error_msg = 'Error loading plugin: ' . $e->getMessage();
    }
}

try {
    $pdo = db();
    $stmt = $pdo->query("SELECT * FROM " . PLUGINS_TABLE . " ORDER BY created_at DESC");
    $plugins = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $error_msg = 'Error loading plugins: ' . $e->getMessage();
}
?>
<!DOCTYPE html>
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=utf-8">
<title>Admin - Manage Plugins</title>
<style>
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

html {
    scroll-behavior: smooth;
}

body {
    background: #0b0b0b;
    color: #eee;
    font-family: Arial, Helvetica, sans-serif;
    margin: 0;
    padding: 20px;
}

.wrap {
    max-width: 1000px;
    margin: 0 auto;
    padding: 20px;
    background: #111;
    border: 1px solid #222;
}

h3 {
    margin: 0 0 20px;
    text-align: center;
    color: #8bc34a;
    font-size: 24px;
    font-weight: bold;
    text-transform: uppercase;
}

.add-form {
    margin-bottom: 30px;
    padding: 20px;
    background: #181818;
    border: 1px solid #333;
}

label {
    display: block;
    margin: 10px 0 5px;
    color: #ccc;
    font-size: 14px;
    font-weight: bold;
}

input[type="text"],
input[type="url"],
textarea {
    width: 100%;
    padding: 10px;
    background: #0b0b0b;
    border: 2px solid #8bc34a;
    color: #eee;
    font-size: 14px;
    font-family: Arial, sans-serif;
}

input:focus, textarea:focus {
    border-color: #4caf50;
    outline: none;
}

textarea {
    min-height: 100px;
    resize: vertical;
}

button {
    padding: 12px 24px;
    background: #8bc34a;
    border: 2px solid #4caf50;
    color: #fff;
    font-size: 14px;
    font-weight: bold;
    cursor: pointer;
    text-transform: uppercase;
    margin-top: 15px;
    width: 100%;
}

button:hover {
    background: #4caf50;
}

.msg {
    padding: 12px;
    margin: 15px 0;
    border-radius: 4px;
    text-align: center;
    font-weight: bold;
}

.msg.success {
    background: rgba(139, 195, 74, 0.2);
    border: 1px solid #8bc34a;
    color: #8bc34a;
}

.msg.error {
    background: rgba(244, 67, 54, 0.2);
    border: 1px solid #f44336;
    color: #f44336;
}

.plugins-list {
    margin-top: 30px;
}

.plugins-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
    gap: 20px;
    width: 100%;
}

.plugin-card {
    background: #181818;
    border: 1px solid #333;
    padding: 15px;
    display: flex;
    flex-direction: column;
    gap: 12px;
    transition: border-color 0.3s, transform 0.3s;
}

.plugin-card:hover {
    border-color: #4caf50;
    transform: translateY(-4px);
}

.plugin-card.editing {
    border-color: #8bc34a;
    box-shadow: 0 0 12px rgba(139, 195, 74, 0.4);
}

.plugin-card.inactive {
    opacity: 0.6;
}

.plugin-thumb {
    width: 100%;
    height: 150px;
    border: 2px solid #8bc34a;
    overflow: hidden;
    display: flex;
    justify-content: center;
    align-items: center;
    background: #0b0b0b;
}

.plugin-thumb img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.plugin-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 10px;
}

.plugin-card-name {
    color: #8bc34a;
    font-weight: bold;
    font-size: 15px;
}

.status-badge {
    font-size: 11px;
    text-transform: uppercase;
    padding: 4px 8px;
    border-radius: 20px;
    border: 1px solid currentColor;
}

.status-badge.active {
    color: #8bc34a;
}

.status-badge.inactive {
    color: #f44336;
}

.plugin-card-desc {
    color: #8bc34a;
    font-size: 13px;
    line-height: 1.4;
    min-height: 40px;
}

.plugin-card-actions {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}

.btn-edit, .btn-toggle, .btn-delete {
    padding: 8px 16px;
    font-size: 12px;
    width: auto;
    margin: 0;
}

.btn-edit {
    background: #2196F3;
    border-color: #1976D2;
}

.btn-toggle {
    background: #FF9800;
    border-color: #F57C00;
}

.btn-delete {
    background: #f44336;
    border-color: #d32f2f;
}

.btn-edit:hover {
    background: #1976D2;
}

.btn-toggle:hover {
    background: #F57C00;
}

.btn-delete:hover {
    background: #d32f2f;
}
</style>
</head>
<body>
<div class="wrap">
  <h3>Admin - Manage Plugins</h3>
  
  <!-- Add/Edit Plugin Form -->
  <div class="add-form">
    <?php if ($success_msg): ?>
      <div class="msg success"><?= h($success_msg) ?></div>
    <?php endif; ?>
    <?php if ($error_msg): ?>
      <div class="msg error"><?= h($error_msg) ?></div>
    <?php endif; ?>
    
    <form method="post">
      <input type="hidden" name="action" value="<?= $editing_plugin ? 'update_plugin' : 'add_plugin' ?>">
      <?php if ($editing_plugin): ?>
        <input type="hidden" name="id" value="<?= $editing_plugin['id'] ?>">
      <?php endif; ?>
      
      <label>Plugin Name:</label>
      <input type="text" name="name" required value="<?= $editing_plugin ? h($editing_plugin['name']) : '' ?>" placeholder="e.g., Advanced Admin Menu">
      
      <label>Image URL (thumbnail):</label>
      <input type="url" name="image_url" value="<?= $editing_plugin ? h($editing_plugin['image_url']) : '' ?>" placeholder="https://example.com/plugin.png">
      
      <label>Description:</label>
      <textarea name="description" placeholder="Plugin description..."><?= $editing_plugin ? h($editing_plugin['description']) : '' ?></textarea>
      
      <label>Installation Tutorial:</label>
      <textarea name="installation_tutorial" placeholder="Step-by-step installation instructions..."><?= $editing_plugin ? h($editing_plugin['installation_tutorial']) : '' ?></textarea>
      
      <label>Video URL (local or external):</label>
      <input type="url" name="video_url" value="<?= $editing_plugin ? h($editing_plugin['video_url']) : '' ?>" placeholder="https://example.com/video.mp4 or /videos/plugin.mp4">
      
      <label>Gallery Images (comma-separated URLs):</label>
      <input type="text" name="gallery_images" value="<?= $editing_plugin ? h($editing_plugin['gallery_images']) : '' ?>" placeholder="https://example.com/img1.jpg, https://example.com/img2.jpg">

      <label>Download Link (Mega / external):</label>
      <input type="url" name="download_url" value="<?= $editing_plugin ? h($editing_plugin['download_url']) : '' ?>" placeholder="https://mega.nz/your-plugin-download">
      
      <button type="submit"><?= $editing_plugin ? 'Update Plugin' : 'Add Plugin' ?></button>
      <?php if ($editing_plugin): ?>
        <a href="admin_plugins.php" style="display: block; text-align: center; margin-top: 10px; color: #8bc34a; text-decoration: none;">Cancel</a>
      <?php endif; ?>
    </form>
  </div>
  
  <!-- Existing Plugins List -->
  <div class="plugins-list">
    <h3 style="margin-bottom: 15px; font-size: 20px;">Existing Plugins</h3>
    <?php if (empty($plugins)): ?>
      <p style="text-align: center; color: #666;">No plugins added yet.</p>
    <?php else: ?>
      <div class="plugins-grid">
        <?php
          $editing_id = $editing_plugin['id'] ?? 0;
          foreach ($plugins as $plugin):
        ?>
          <?php
            $desc_preview = '';
            if (!empty($plugin['description'])) {
                $source_desc = $plugin['description'];
                if (function_exists('mb_strlen')) {
                    $desc_preview = mb_strlen($source_desc, 'UTF-8') > 140
                        ? mb_substr($source_desc, 0, 140, 'UTF-8') . '...'
                        : $source_desc;
                } else {
                    $desc_preview = strlen($source_desc) > 140
                        ? substr($source_desc, 0, 140) . '...'
                        : $source_desc;
                }
            }
          ?>
          <div class="plugin-card <?= !$plugin['active'] ? 'inactive' : '' ?> <?= ($editing_id === intval($plugin['id'])) ? 'editing' : '' ?>">
            <div class="plugin-thumb">
              <img src="<?= h($plugin['image_url']) ?: 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'300\' height=\'200\'%3E%3Crect fill=\'%23333\' width=\'300\' height=\'200\'/%3E%3Ctext x=\'50%25\' y=\'50%25\' text-anchor=\'middle\' dy=\'.3em\' fill=\'%23666\' font-size=\'14\'%3ENo Image%3C/text%3E%3C/svg%3E' ?>" alt="<?= h($plugin['name']) ?>">
            </div>
            <div class="plugin-card-header">
              <div class="plugin-card-name"><?= h($plugin['name']) ?></div>
              <span class="status-badge <?= $plugin['active'] ? 'active' : 'inactive' ?>"><?= $plugin['active'] ? 'Active' : 'Inactive' ?></span>
            </div>
            <?php if ($desc_preview): ?>
              <div class="plugin-card-desc"><?= h($desc_preview) ?></div>
            <?php endif; ?>
            <div class="plugin-card-actions">
              <a href="?edit=<?= $plugin['id'] ?>" class="btn-edit" style="text-decoration: none; display: inline-block;">Edit</a>
              <a href="?toggle=<?= $plugin['id'] ?>" class="btn-toggle" style="text-decoration: none; display: inline-block;"><?= $plugin['active'] ? 'Deactivate' : 'Activate' ?></a>
              <a href="?delete=<?= $plugin['id'] ?>" class="btn-delete" style="text-decoration: none; display: inline-block;" onclick="return confirm('Are you sure you want to delete this plugin?')">Delete</a>
            </div>
          </div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>
  </div>
  
  <div style="text-align: center; margin-top: 30px;">
    <a href="plugins_gallery.php" style="color: #8bc34a; text-decoration: none; font-size: 14px;">View Public Gallery →</a>
  </div>
</div>
</body>
</html>

